/* Module to process system calls from emulated Intel code.
 *
 * The Linux/Intel system call convention is as follows:
 *
 * Upon entry into the kernel, %eax contains the system call number.
 *
 * Arguments are carried in the registers:
 *
 *	%ebx, %ecx, %edx, %esi, %edi, %ebp
 *
 * Return value is carried in %eax: negative if error, >= 0 if success.
 *
 */

#include <stdio.h>
#include <errno.h>
#include <unistd.h>

#include <sys/syslog.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/swap.h>
#include <sys/vfs.h>

#include <syscall_thunk.h>

extern FILE * x86_logfile;


/* Forward declarations for system-call handling routines */
int	do_newstat(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_newfstat(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_newlstat(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_oldstat(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_oldfstat(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_oldlstat(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_brk(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_exit(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_old_mmap(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_ioctl(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_open(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_getrusage(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_time(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_personality(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_execve(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_stime(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_utime(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_ftime(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_times(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_olduname(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_uname(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_newuname(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_llseek(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_sigaction(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_fcntl(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_getdents(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_old_select(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_signal(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_sigprocmask(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_ssetmask(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_sgetmask(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_sigsuspend(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_sigpending(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_getrlimit(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_setrlimit(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_getgroups(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_setgroups(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_socketcall(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_readv(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_writev(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_mprotect(int a0, int a1, int a2, int a3, int a4, int a5);
int	do_wait4(int a0, int a1, int a2, int a3, int a4, int a5);

/* Syscalls not declared in include files... */
int	waitpid(int, int *, int);
int	creat(char *, int);
int	nice(int);
int	kill(int, int);
int	reboot(int, int, int);
int	getpgid(int);
int	setfsuid(int);
int	setfsgid(int);
int	getsid(void);
int	ustat(int, struct ustat *);

typedef enum {
	TK_UNIMP,		/* Unimplemented by Linux */
	TK_UNIMP_FATAL,		/* Unimplemented by syscall_thunk (and should be) */
	TK_PASSTHROUGH,		/* Pass this syscall straight through */
	TK_THUNK		/* Call a thunking routine */
} thunk_type;

/* Other classes of unimplemented syscalls, defined here for human consumption */
#define TK_LATER	TK_UNIMP_FATAL		/* Deferred implementation */
#define TK_NEVER	TK_UNIMP_FATAL		/* We'll NEVER support this one! */

typedef enum {
	SC_UNUSED,		/* Argument is unused */
	SC_INT,			/* Argument is an integer */
	SC_PTR,			/* Argument is a pointer */
	SC_STRING,		/* Argument is a string pointer */
	SC_SPECIAL		/* Argument requires special processing */
} sc_argtype;

struct syscall_descriptor {
	thunk_type	type;
	int		(*routine)();
	char		*name;
	sc_argtype	argtypes[6];
	int		flags;
};

/* For now there's only one bit for the flags word, but we may want
 * more later...
 */
#define SD_FLAGS_TRACE		1	/* Trace this syscall */

/* This flag exists for the benefit of external (thunking) routines */
int	x86_trace_current_syscall = 0;

/* Table of routines for thunking system calls */
struct syscall_descriptor	syscall_table[] = {

	{ TK_NEVER, NULL, "setup", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_exit, "exit", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fork, "fork", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, read, "read", { SC_INT, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, write, "write", { SC_INT, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_open, "open", { SC_STRING, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, close, "close", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, waitpid, "waitpid", { SC_INT, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, creat, "creat", { SC_STRING, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, link, "link", { SC_STRING, SC_STRING, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, unlink, "unlink", { SC_STRING, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_execve, "execve", { SC_STRING, SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, chdir, "chdir", { SC_STRING, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_time, "time", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, mknod, "mknod", { SC_STRING, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, chmod, "chmod", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, chown, "chown", { SC_PTR, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "break", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_oldstat, "stat", { SC_STRING, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, lseek, "lseek", { SC_INT, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getpid, "getpid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "mount", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "umount", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setuid, "setuid", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getuid, "getuid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_stime, "stime", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "ptrace", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, alarm, "alarm", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_oldfstat, "fstat", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, pause, "pause", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_utime, "utime", { SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "stty", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "gtty", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, access, "access", { SC_STRING, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, nice, "nice", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_ftime, "ftime", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, sync, "sync", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, kill, "kill", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, rename, "rename", { SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, mkdir, "mkdir", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, rmdir, "rmdir", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, dup, "dup", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, pipe, "pipe", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_times, "times", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "prof", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_brk, "brk", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setgid, "setgid", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getgid, "getgid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_signal, "signal", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, geteuid, "geteuid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getegid, "getegid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "acct", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "phys", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "lock", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_ioctl, "ioctl", { SC_INT, SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_fcntl, "fcntl", { SC_INT, SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "mpx", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setpgid, "setpgid", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "ulimit", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_olduname, "olduname", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, umask, "umask", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, chroot, "chroot", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, ustat, "ustat", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, dup2, "dup2", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getppid, "getppid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getpgrp, "getpgrp", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setsid, "setsid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_sigaction, "sigaction", { SC_INT, SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_sgetmask, "sgetmask", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_ssetmask, "ssetmask", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setreuid, "setreuid", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setregid, "setregid", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_sigsuspend, "sigsuspend", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_sigpending, "sigpending", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, sethostname, "sethostname", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_setrlimit, "setrlimit", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_getrlimit, "getrlimit", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_getrusage, "getrusage", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, gettimeofday, "gettimeofday", { SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, settimeofday, "settimeofday", { SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_getgroups, "getgroups", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_setgroups, "setgroups", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_old_select, "old_select", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, symlink, "symlink", { SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_oldlstat, "lstat", { SC_STRING, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, readlink, "readlink", { SC_PTR, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "uselib", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, swapon, "swapon", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, reboot, "reboot", { SC_INT, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "old_readdir", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_old_mmap, "old_mmap", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, munmap, "munmap", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, truncate, "truncate", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, ftruncate, "ftruncate", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fchmod, "fchmod", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fchown, "fchown", { SC_INT, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getpriority, "getpriority", { SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setpriority, "setpriority", { SC_INT, SC_INT, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "profil", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, statfs, "statfs", { SC_STRING, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fstatfs, "fstatfs", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "ioperm", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_socketcall, "socketcall", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, syslog, "syslog", { SC_INT, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setitimer, "setitimer", { SC_INT, SC_PTR, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getitimer, "getitimer", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_newstat, "newstat", { SC_STRING, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_newlstat, "newlstat", { SC_STRING, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_newfstat, "newfstat", { SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_uname, "uname", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "iopl", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, vhangup, "vhangup", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "idle", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "vm86", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_wait4, "wait4", { SC_INT, SC_PTR, SC_INT, SC_PTR, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, swapoff, "swapoff", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sysinfo", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "ipc", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fsync, "fsync", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sigreturn", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "clone", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setdomainname, "setdomainname", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_newuname, "newuname", { SC_PTR, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "modify_ldt", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "adjtimex", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_mprotect, "mprotect", { SC_PTR, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_sigprocmask, "sigprocmask", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "create_module", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "init_module", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "delete_module", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "get_kernel_syms", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "quotactl", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getpgid, "getpgid", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fchdir, "fchdir", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "bdflush", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sysfs", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_personality, "personality", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "afs_syscall", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setfsuid, "setfsuid", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, setfsgid, "setfsgid", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_llseek, "llseek", { SC_INT, SC_INT, SC_INT, SC_PTR, SC_INT, SC_UNUSED, 0 } },
	{ TK_THUNK, do_getdents, "getdents", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, select, "select", { SC_INT, SC_PTR, SC_PTR, SC_PTR, SC_PTR, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "flock", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "msync", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_readv, "readv", { SC_INT, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_THUNK, do_writev, "writev", { SC_INT, SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, getsid, "getsid", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_PASSTHROUGH, fdatasync, "fdatasync", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_NEVER, NULL, "sysctl", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "mlock", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "munlock", { SC_PTR, SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "mlockall", { SC_INT, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "munlockall", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_setparam", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_getparam", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_setscheduler", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_getscheduler", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_yield", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_get_priority_max", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_get_priority_min", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "sched_rr_get_interval", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "nanosleep", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } },
	{ TK_LATER, NULL, "mremap", { SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, SC_UNUSED, 0 } }

};

#define NUM_SYSCALLS 164

/* Utility function: Table-driven functions to translate bitmasks
 * between X86 and Alpha formats...
 */
unsigned int
x86_to_alpha_bitmask(unsigned int x86_mask, 
		     bitmask_transtbl * trans_tbl)
{
    bitmask_transtbl *	btp;
    unsigned int	alpha_mask = 0;

    for(btp = trans_tbl; btp->x86_mask && btp->alpha_mask; btp++) {
	if((x86_mask & btp->x86_mask) == btp->x86_bits) {
	    alpha_mask |= btp->alpha_bits;
	}
    }
    return(alpha_mask);
}

unsigned int
alpha_to_x86_bitmask(unsigned int alpha_mask, 
		     bitmask_transtbl * trans_tbl)
{
    bitmask_transtbl *	btp;
    unsigned int	x86_mask = 0;

    for(btp = trans_tbl; btp->x86_mask && btp->alpha_mask; btp++) {
	if((alpha_mask & btp->alpha_mask) == btp->alpha_bits) {
	    x86_mask |= btp->x86_mask;
	}
    }
    return(x86_mask);
}

/* Utility function: Table-driven functions to translate parameters
 * between X86 and Alpha formats...
 */
unsigned int
x86_to_alpha_param(unsigned int x86_param, 
		     param_transtbl * trans_tbl, int table_size)
{
    int i;
    param_transtbl *	ptp;

    for(i = 0, ptp = trans_tbl; i < table_size; i++, ptp++) {
	if(x86_param == ptp->x86_param) {
	    return(ptp->alpha_param);
	}
    }
    fprintf(x86_logfile, "Unrecognized parameter %d\n", x86_param);
    return(-1);
}

unsigned int
alpha_to_x86_param(unsigned int alpha_param, 
		     param_transtbl * trans_tbl, int table_size)
{
    int i;
    param_transtbl *	ptp;

    for(i = 0, ptp = trans_tbl; i < table_size; i++, ptp++) {
	if(alpha_param == ptp->alpha_param) {
	    return(ptp->x86_param);
	}
    }
    fprintf(x86_logfile, "Unrecognized parameter %d\n", alpha_param);
    return(-1);
}


/* Translations for error codes (Linux/Intel and Linux/Alpha, it turns out, have different
 * sets of error codes... grrr...)
 */

static int intel_to_alpha_errno_table[] = {
	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 35, 12, 13, 14, 15, 
	16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 
	32, 33, 34, 11, 63, 77, 78, 66, 62, 0, 80, 81, 88, 89, 90, 91, 
	93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 0, 104, 87, 86, 83, 82, 
	105, 92, 71, 106, 107, 108, 109, 85, 110, 111, 84, 112, 113, 114, 115, 122, 
	123, 124, 125, 126, 116, 127, 128, 68, 38, 39, 40, 41, 42, 43, 44, 45, 
	46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 
	64, 65, 37, 36, 70, 117, 118, 119, 120, 121, 69, 0, 0, 0, 0, 0
};

static int alpha_to_intel_errno_table[] = {
	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 35, 12, 13, 14, 15, 
	16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 
	32, 33, 34, 11, 115, 114, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 
	98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 40, 36, 
	112, 113, 39, 0, 87, 122, 116, 66, 0, 0, 0, 0, 0, 37, 38, 0, 
	42, 43, 63, 62, 74, 71, 61, 60, 44, 45, 46, 47, 65, 48, 49, 50, 
	51, 52, 53, 54, 55, 56, 57, 0, 59, 64, 67, 68, 69, 70, 72, 73, 
	75, 76, 77, 78, 84, 117, 118, 119, 120, 121, 79, 80, 81, 82, 83, 85
};

int intel_to_alpha_errno(int e)
{
    if(e < 0 || e > 127) {
	return(E2BIG);
    }
    else {
        return(intel_to_alpha_errno_table[e]);
    }
}

int alpha_to_intel_errno(int e)
{
    if(e < 0 || e > 127) {
	return(E2BIG);
    }
    else {
        return(alpha_to_intel_errno_table[e]);
    }
}


int do_syscall(int eax, int ebx, int ecx, int edx, int esi, int edi, int ebp)
{
    struct syscall_descriptor *	sd;
    int i;
    
    if(eax >= 0 && eax < NUM_SYSCALLS) {
        sd = &(syscall_table[eax]);

	if(sd->flags & SD_FLAGS_TRACE) {
	    int args[6];
	    x86_trace_current_syscall = 1;
	    args[0] = ebx; args[1] = ecx; args[2] = edx;
	    args[3] = esi; args[4] = edi; args[5] = ebp;

	    fprintf(x86_logfile, "[%d] SYSCALL: %s(", getpid(), sd->name);
	    for(i = 0; i < 6; i++) {
		switch(sd->argtypes[i]) {
		    case SC_INT:
		        fprintf(x86_logfile, "%d ", args[i]);
			break;
		    case SC_PTR:
			fprintf(x86_logfile, "0x%x ", args[i]);
			break;
		    case SC_STRING:
			fprintf(x86_logfile, "\"%s\" ", 
				 (char *)((unsigned long)args[i] & 0xffffffff));
			break;
		}
	    }
	    fprintf(x86_logfile, ")\r\n");
	    fflush(x86_logfile);
	}

	if((sd->type == TK_UNIMP) ||
	   (sd->type == TK_UNIMP_FATAL)) {
	    fprintf(x86_logfile, 
	      "Unimplemented system call: %s(0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x)\n", 
	      		sd->name, ebx, ecx, edx, esi, edi, ebp);
	    if(sd->type == TK_UNIMP_FATAL) {
	    	em86_exit(-1);
		exit(-1);
	    }
	    else {
		if(sd->flags & SD_FLAGS_TRACE) {
		    fprintf(x86_logfile, "   ....returns 0x%x\r\n", -ENOSYS);
		    fflush(x86_logfile);
		}
		return(-ENOSYS);
	    }
	}


	if(sd->type == TK_THUNK) {
	    eax = (sd->routine)(ebx, ecx, edx, esi, edi, ebp);
	    if(sd->flags & SD_FLAGS_TRACE) {
	        fprintf(x86_logfile, "   ....returns 0x%x\r\n", eax);
		fflush(x86_logfile);
	    }
	    return(eax);
	}

	if(sd->type == TK_PASSTHROUGH) {
	    unsigned long	aa0, aa1, aa2, aa3, aa4, aa5;
	    long		retval;

	    /* Massage args if necessary and pass this through to a straight system call... */
	    if((sd->argtypes[0] == SC_PTR) || (sd->argtypes[0] == SC_STRING)) {
		aa0 = (unsigned long)(ebx & 0xffffffff);
	    }
	    else {
	    	aa0 = ebx;
	    }
	    if((sd->argtypes[1] == SC_PTR) || (sd->argtypes[1] == SC_STRING)) {
		aa1 = (unsigned long)(ecx & 0xffffffff);
	    }
	    else {
	    	aa1 = ecx;
	    }
	    if((sd->argtypes[2] == SC_PTR) || (sd->argtypes[2] == SC_STRING)) {
		aa2 = (unsigned long)(edx & 0xffffffff);
	    }
	    else {
	    	aa2 = edx;
	    }
	    if((sd->argtypes[3] == SC_PTR) || (sd->argtypes[3] == SC_STRING)) {
		aa3 = (unsigned long)(esi & 0xffffffff);
	    }
	    else {
		aa3 = esi;
	    }
	    if((sd->argtypes[4] == SC_PTR) || (sd->argtypes[5] == SC_STRING)) {
		aa4 = (unsigned long)(edi & 0xffffffff);
	    }
	    else {
	    	aa4 = edi;
	    }
	    if((sd->argtypes[5] == SC_PTR) || (sd->argtypes[6] == SC_STRING)) {
		aa5 = (unsigned long)(ebp & 0xffffffff);
	    }
	    else {
	    	aa5 = ebp;
	    }

	    retval = (long)((sd->routine)(aa0, aa1, aa2, aa3, aa4, aa5));
	    if(retval == -1) {
		if(sd->flags & SD_FLAGS_TRACE) {
		    fprintf(x86_logfile, "   ....returns 0x%x\r\n", -alpha_to_intel_errno(errno));
		    fflush(x86_logfile);
		}
	    	return(-alpha_to_intel_errno(errno));
	    }
	    else {
		if(sd->flags & SD_FLAGS_TRACE) {
		    fprintf(x86_logfile, "   ....returns 0x%x\r\n", retval);
		    fflush(x86_logfile);
		}
		return(retval);
	    }

	}
	x86_trace_current_syscall = 0;
    }
    else {
	fprintf(x86_logfile, "Undefined system call %d\n", eax);
	return(-1);
    }

    return(eax);
}


void	x86_syscall_trace(char * name) 
{
    int		i;

    for(i = 0; i < NUM_SYSCALLS; i++) {
    	if(strcmp(name, syscall_table[i].name) == 0) {
	    syscall_table[i].flags |= SD_FLAGS_TRACE;
	    return;
	}
    }

    fprintf(x86_logfile, "Trace request for unknown syscall '%s'\n", name);
}

void	x86_syscall_trace_all()
{
    int		i;

    for(i = 0; i < NUM_SYSCALLS; i++) {
	    syscall_table[i].flags |= SD_FLAGS_TRACE;
    }
}

void	x86_syscall_untrace(char * name) 
{
    int		i;

    for(i = 0; i < NUM_SYSCALLS; i++) {
    	if(strcmp(name, syscall_table[i].name) == 0) {
	    syscall_table[i].flags &= ~SD_FLAGS_TRACE;
	    return;
	}
    }

    fprintf(x86_logfile, "Un-Trace request for unknown syscall '%s'\n", name);
}

