/* This module keeps track of memory pages in the emulated program.
 * We have to do this because, unfortunately, the x86 has a 4k page
 * granularity while Alpha has 8K page granularity.  This means that
 * WE have to keep track of 4k pages because the system can't.
 *
 * We use one byte per page to maintain various flags, and we
 * implement this map as a sparse array to save space.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define X86_PAGE_SIZE	0x1000
#define X86_PAGE_SHIFT	12
#define ALPHA_PAGE_SIZE	0x2000
#define ALHPA_PAGE_SHIFT 13

#define	MAX_ADDR	0x80000000	/* Highest address we keep track of */
#define DIRECTORY_SIZE	0x200		/* Number of entries in the page
					 * directory (first level of the
					 * sparse array)
					 */
#define DIRECTORY_SHIFT	9

#define ENTRIES_PER_TABLE	(MAX_ADDR>>(DIRECTORY_SHIFT+X86_PAGE_SHIFT))

#define table_entry	unsigned char	

static table_entry *	page_directory[DIRECTORY_SIZE];
static int		page_directory_initialized = 0;

extern int debug_mmap;
extern FILE *x86_logfile;

static void page_map_init()
{
    int		i;

    for(i = 0; i < DIRECTORY_SIZE; i++) {
	page_directory[i] = NULL;
    }
    page_directory_initialized = 1;
}

void put_page_bits(unsigned long addr, table_entry bits)
{
    int		pagenum;
    int		tablenum;
    int		entrynum;
    table_entry *table;

#if 0
    if(debug_mmap) {
	fprintf(x86_logfile, "MMAP: Setting page bits for addr 0x%x: [%s]\n",
		addr, print_prot(bits));
    }
#endif

    if(!page_directory_initialized) {
	page_map_init();
    }

    if(addr > MAX_ADDR) {
	/* We don't keep track of these addresses... */
	return;
    }

    pagenum = addr >> X86_PAGE_SHIFT;
    tablenum = pagenum / ENTRIES_PER_TABLE;
    entrynum = pagenum & (ENTRIES_PER_TABLE-1);

    table = page_directory[tablenum];
    if(table == NULL) {
	/* Nothing here; need to allocate a table */
	table = (table_entry *)malloc(ENTRIES_PER_TABLE*sizeof(table_entry));
	memset(table, 0, ENTRIES_PER_TABLE*sizeof(table_entry));
	page_directory[tablenum] = table;
    }

    table[entrynum] = bits;
}


table_entry get_page_bits(unsigned long addr)
{
    int		pagenum;
    int		tablenum;
    int		entrynum;
    table_entry *table;

    if(!page_directory_initialized) {
	page_map_init();
    }

    if(addr > MAX_ADDR) {
	fprintf(x86_logfile, "Address map overflow: 0x%lx\n", addr);
	exit(-1);
    }

    pagenum = addr >> X86_PAGE_SHIFT;
    tablenum = pagenum / ENTRIES_PER_TABLE;
    entrynum = pagenum & (ENTRIES_PER_TABLE-1);

    table = page_directory[tablenum];
    if(table == NULL) {
	/* Nothing here; page must be invalid */
	return(0);
    }

    return(table[entrynum]);
}

/* Set page bits for a range of pages */
void set_page_range_bits(unsigned long addr, unsigned long end, table_entry bits)
{
    long	len = end - addr + 1;

    if(addr > MAX_ADDR) {
	/* We don't keep track of these addresses... */
	return;
    }
    while(len > 0) {
	put_page_bits(addr, bits);
	addr += X86_PAGE_SIZE;
	len -= X86_PAGE_SIZE;
    }
}
